\
<?php
declare(strict_types=1);

error_reporting(E_ALL);
ini_set('display_errors', '1');

function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

$root = realpath(__DIR__ . '/../../'); // cms/
$lock = $root . '/storage/app/install.lock';

$step = $_POST['step'] ?? ($_GET['step'] ?? 'welcome');

// Basic lock check (after Laravel exists)
if (file_exists($lock) && $step !== 'unlock') {
    http_response_code(403);
    echo "<h2>Installer locked</h2><p>Delete <code>cms/storage/app/install.lock</code> to re-run.</p>";
    exit;
}

function run_cmd(string $cmd, string $cwd, array &$log): bool {
    $log[] = ">>> $cmd";
    $descriptorspec = [1 => ["pipe","w"], 2 => ["pipe","w"]];
    $process = proc_open($cmd, $descriptorspec, $pipes, $cwd);
    if (!is_resource($process)) { $log[] = "proc_open failed"; return false; }
    $stdout = stream_get_contents($pipes[1]); fclose($pipes[1]);
    $stderr = stream_get_contents($pipes[2]); fclose($pipes[2]);
    $code = proc_close($process);
    if ($stdout) $log[] = $stdout;
    if ($stderr) $log[] = $stderr;
    $log[] = "exit code: $code";
    return $code === 0;
}

function ensure_dir($p){ if(!is_dir($p)) @mkdir($p, 0775, true); }

$log = [];
$msg = "";

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if ($step === 'install') {
        $db_name = trim($_POST['db_name'] ?? '');
        $db_user = trim($_POST['db_user'] ?? '');
        $db_pass = (string)($_POST['db_pass'] ?? '');
        $db_host = trim($_POST['db_host'] ?? 'localhost');
        $db_port = trim($_POST['db_port'] ?? '3306');
        $app_url = trim($_POST['app_url'] ?? '');
        $site_root = trim($_POST['site_root'] ?? '');

        // Validate
        if (!$db_name || !$db_user || !$app_url) {
            $msg = "Missing required fields.";
        } else {
            // 1) Download composer.phar if missing
            if (!file_exists($root . '/composer.phar')) {
                $composerUrl = 'https://getcomposer.org/download/latest-stable/composer.phar';
                $data = @file_get_contents($composerUrl);
                if ($data === false) {
                    $msg = "Failed to download composer.phar. Check allow_url_fopen or outbound access.";
                } else {
                    file_put_contents($root . '/composer.phar', $data);
                }
            }

            if (!$msg) {
                // 2) Create Laravel project if artisan missing
                if (!file_exists($root . '/artisan')) {
                    // Create in temp then move (to avoid nested folder)
                    $tmp = $root . '/_laravel_tmp';
                    if (is_dir($tmp)) { /* ignore */ }
                    ensure_dir($tmp);
                    $ok = run_cmd("php composer.phar create-project laravel/laravel .", $tmp, $log);
                    if (!$ok) {
                        $msg = "Failed creating Laravel project.";
                    } else {
                        // Move files from tmp into cms root
                        $it = new RecursiveIteratorIterator(
                            new RecursiveDirectoryIterator($tmp, FilesystemIterator::SKIP_DOTS),
                            RecursiveIteratorIterator::SELF_FIRST
                        );
                        foreach ($it as $file) {
                            $dest = $root . '/' . $it->getSubPathName();
                            if ($file->isDir()) {
                                ensure_dir($dest);
                            } else {
                                ensure_dir(dirname($dest));
                                copy((string)$file, $dest);
                            }
                        }
                        // cleanup tmp
                        // (best-effort)
                    }
                }
            }

            if (!$msg) {
                // 3) Write .env
                if (!file_exists($root . '/.env')) {
                    copy($root . '/.env.example', $root . '/.env');
                }
                $env = file_get_contents($root . '/.env');
                $repl = [
                    '/^APP_URL=.*/m' => 'APP_URL=' . $app_url,
                    '/^APP_ENV=.*/m' => 'APP_ENV=production',
                    '/^APP_DEBUG=.*/m' => 'APP_DEBUG=false',
                    '/^DB_CONNECTION=.*/m' => 'DB_CONNECTION=mysql',
                    '/^DB_HOST=.*/m' => 'DB_HOST=' . $db_host,
                    '/^DB_PORT=.*/m' => 'DB_PORT=' . $db_port,
                    '/^DB_DATABASE=.*/m' => 'DB_DATABASE=' . $db_name,
                    '/^DB_USERNAME=.*/m' => 'DB_USERNAME=' . $db_user,
                    '/^DB_PASSWORD=.*/m' => 'DB_PASSWORD=' . $db_pass,
                    '/^CACHE_STORE=.*/m' => 'CACHE_STORE=file',
                    '/^SESSION_DRIVER=.*/m' => 'SESSION_DRIVER=file',
                    '/^QUEUE_CONNECTION=.*/m' => 'QUEUE_CONNECTION=sync',
                ];
                foreach ($repl as $p => $v) {
                    if (preg_match($p, $env)) $env = preg_replace($p, $v, $env);
                    else $env .= "\n" . $v . "\n";
                }
                file_put_contents($root . '/.env', $env);

                // 4) Install PHP packages
                $ok = run_cmd("php composer.phar install --no-dev --optimize-autoloader", $root, $log);
                if (!$ok) $msg = "Composer install failed.";
            }

            if (!$msg) {
                // 5) Require Filament + Sanctum (if not present)
                $ok = run_cmd("php composer.phar require filament/filament:^3.2 laravel/sanctum:^4.0", $root, $log);
                if (!$ok) $msg = "Composer require failed (Filament/Sanctum).";
            }

            if (!$msg) {
                // 6) Artisan key + migrate
                $ok = run_cmd("php artisan key:generate --force", $root, $log);
                if (!$ok) $msg = "key:generate failed.";

                $ok = run_cmd("php artisan sanctum:install --force", $root, $log);
                // ignore errors here sometimes

                $ok = run_cmd("php artisan migrate --force", $root, $log);
                if (!$ok) $msg = "migrate failed.";
            }

            if (!$msg) {
                // 7) Copy stubs (API, models, migrations, Filament resources)
                $stubRoot = realpath(__DIR__ . '/../../install_stubs');
                if (!$stubRoot) {
                    $msg = "install_stubs missing.";
                } else {
                    $rii = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($stubRoot, FilesystemIterator::SKIP_DOTS));
                    foreach ($rii as $file) {
                        $rel = str_replace($stubRoot . DIRECTORY_SEPARATOR, '', (string)$file);
                        $dest = $root . DIRECTORY_SEPARATOR . $rel;
                        if (!file_exists($dest)) {
                            @mkdir(dirname($dest), 0775, true);
                            copy((string)$file, $dest);
                        }
                    }
                }
            }

            if (!$msg) {
// 8) Register command (safe): use Artisan's auto-discovery (Laravel 8+)
// We add the command via routes/console.php to avoid editing Kernel.php with regex/backslashes.
$consoleRoutes = $root . '/routes/console.php';
if (file_exists($consoleRoutes)) {
    $c = file_get_contents($consoleRoutes);
    if (strpos($c, 'RemarkBootstrap') === false) {
        $c .= "\n\nArtisan::command('remark:bootstrap', function () {\n    $this->call(\\App\\Console\\Commands\\RemarkBootstrap::class);\n});\n";
        file_put_contents($consoleRoutes, $c);
    }
}

// 9) Ensure Filament panel (default /admin)
// Filament auto-discovers resources, no extra needed.
                // 10) Migrate again for pages/posts
                $ok = run_cmd("php artisan migrate --force", $root, $log);
                if (!$ok) $msg = "migrate (pages/posts) failed.";
            }

            if (!$msg) {
                // 11) Create admin + token
                $ok = run_cmd("php artisan remark:bootstrap --force", $root, $log);
                if (!$ok) $msg = "bootstrap failed.";
            }

            $tokenFile = $root . '/storage/app/remark-token.txt';
            $token = file_exists($tokenFile) ? trim((string)file_get_contents($tokenFile)) : '';

            // 12) Optional: build static site and copy out
            $siteResult = "";
            if ($site_root) {
                $webRoot = realpath($root . '/../web');
                if (!$webRoot || !is_dir($webRoot)) {
                    $siteResult = "Web folder not found at ../web";
                } else {
                    // write web env
                    $envLocal = "NEXT_PUBLIC_CMS_URL={$app_url}\nCMS_API_TOKEN={$token}\n";
                    file_put_contents($webRoot . '/.env.local', $envLocal);

                    $ok = run_cmd("npm install", $webRoot, $log);
                    if ($ok) $ok = run_cmd("npm run build:static", $webRoot, $log);

                    $outDir = $webRoot . '/out';
                    if ($ok && is_dir($outDir)) {
                        // copy out -> site_root
                        $siteRoot = rtrim($site_root, '/');
                        if (!is_dir($siteRoot)) @mkdir($siteRoot, 0775, true);

                        $it = new RecursiveIteratorIterator(
                            new RecursiveDirectoryIterator($outDir, FilesystemIterator::SKIP_DOTS),
                            RecursiveIteratorIterator::SELF_FIRST
                        );
                        foreach ($it as $f) {
                            $dest = $siteRoot . '/' . $it->getSubPathName();
                            if ($f->isDir()) {
                                if (!is_dir($dest)) @mkdir($dest, 0775, true);
                            } else {
                                @mkdir(dirname($dest), 0775, true);
                                copy((string)$f, $dest);
                            }
                        }
                        $siteResult = "Static site copied to: " . $siteRoot;
                    } else {
                        $siteResult = "Failed to build static site. Your hosting may block npm from exec(). You can build using cPanel Node App UI instead.";
                    }
                }
            }

            // 13) Lock installer
            ensure_dir($root . '/storage/app');
            file_put_contents($lock, date('c'));

            // Done page
            echo "<h2>✅ Installed</h2>";
            echo "<p><b>CMS Admin:</b> <a href='" . h($app_url) . "/admin'>Open /admin</a></p>";
            echo "<p><b>Admin login:</b> admin@local.test / Admin@12345</p>";
            echo "<p><b>API Token:</b> <code style='word-break:break-all'>" . h($token ?: '(not found)') . "</code></p>";
            echo "<p><b>Site build:</b> " . h($siteResult ?: 'Skipped') . "</p>";
            echo "<hr><h3>Logs</h3><pre style='white-space:pre-wrap'>" . h(implode("
", $log)) . "</pre>";
            echo "<p>Security: installer locked. Delete <code>cms/storage/app/install.lock</code> to re-run.</p>";
            exit;
        }
    }
}

?>
<!doctype html>
<html>
<head>
  <meta charset="utf-8">
  <title>Remark Installer</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <style>
    body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif;margin:0;background:#f6f7fb}
    .wrap{max-width:880px;margin:40px auto;padding:0 16px}
    .card{background:#fff;border:1px solid #e8e9ef;border-radius:16px;padding:20px}
    .row{display:flex;gap:16px;flex-wrap:wrap}
    .col{flex:1;min-width:260px}
    label{display:block;font-size:13px;margin:10px 0 6px;color:#333}
    input{width:100%;padding:10px 12px;border:1px solid #d7d9e5;border-radius:10px}
    button{padding:12px 16px;border:0;border-radius:12px;background:#111;color:#fff;cursor:pointer;margin-top:14px}
    .hint{font-size:13px;color:#555;line-height:1.5}
    .msg{padding:10px 12px;border-radius:10px;background:#fff7d6;border:1px solid #ffe08a;margin-bottom:12px}
    code{background:#f1f2f6;padding:2px 6px;border-radius:6px}
  </style>
</head>
<body>
  <div class="wrap">
    <div class="card">
      <h2>Remark CMS Installer</h2>
      <p class="hint">No terminal needed. This will install Laravel + Filament CMS and (optionally) build the static Next.js site.</p>

      <?php if ($msg): ?><div class="msg"><?=h($msg)?></div><?php endif; ?>

      <form method="post">
        <input type="hidden" name="step" value="install">
        <div class="row">
          <div class="col">
            <h3>CMS Settings</h3>
            <label>APP URL (CMS domain)</label>
            <input name="app_url" placeholder="https://cms.yourdomain.com" required>

            <h3>Database</h3>
            <label>DB Host</label>
            <input name="db_host" value="localhost">
            <label>DB Port</label>
            <input name="db_port" value="3306">
            <label>DB Name</label>
            <input name="db_name" placeholder="cpanel_dbname" required>
            <label>DB Username</label>
            <input name="db_user" placeholder="cpanel_user" required>
            <label>DB Password</label>
            <input type="password" name="db_pass" placeholder="password">
          </div>
          <div class="col">
            <h3>Static Site (Optional)</h3>
            <p class="hint">
              If your hosting allows running <code>npm</code> from PHP exec, the installer can build the static site and copy it.
              If not, you can build using cPanel Node UI.
            </p>
            <label>Site Document Root (copy /web/out here)</label>
            <input name="site_root" placeholder="/home/USER/public_html">

            <p class="hint">
              If you leave this empty, site build will be skipped.
            </p>

            <h3>After install</h3>
            <p class="hint">
              Admin: <code>admin@local.test</code> / <code>Admin@12345</code><br>
              CMS admin URL: <code>/admin</code><br>
              API routes are protected by Bearer token.
            </p>
          </div>
        </div>

        <button type="submit">Install Now</button>
      </form>
    </div>
  </div>
</body>
</html>
